// Written by Joseph Vohnoutka, vohno013
public class Board {

    // Instance variables
    private Piece[][] board;
    // String array of the unicode number characters
    static final String[] unicodeNumbers = {"\uFF10", "\uFF11", "\uFF12", "\uFF13", "\uFF14", "\uFF15", "\uFF16", "\uFF17"};

    public Board() {
        board = new Piece[8][8];
    }

    // Accessor Methods

    public Piece getPiece(int row, int col) {
        // gets the piece from the provided row and column
        return board[row][col];
    }

    public void setPiece(int row, int col, Piece piece) {
        // sets the provided piece to the provided row and column
        board[row][col] = piece;
    }


    // Moves a Piece object from one cell in the board to another, provided that
    // this movement is legal. Returns a boolean to signify success or failure.
    // This method calls all necessary helper functions to determine if a move
    // is legal, and to execute the move if it is.
    public boolean movePiece(int startRow, int startCol, int endRow, int endCol) {
        // gets the piece at the selected position
        Piece selectedPiece = getPiece(startRow, startCol);
        // makes sure there is a valid piece
        if (selectedPiece == null) return false;
        // checks if the move is allowed and changes the location of the piece if it is and returns true. If not, returns false.
        if (selectedPiece.isMoveLegal(this, endRow, endCol)) {
            setPiece(endRow, endCol, selectedPiece);
            setPiece(startRow, startCol, null);
            selectedPiece.setPosition(endRow, endCol);
            return true;
        }
        return false;
    }

    // Determines whether the game has been ended, i.e., if one player's King
    // has been captured.
    public boolean isGameOver() {
        Piece tempPiece = null;
        boolean blackKingAlive = false;
        boolean whiteKingAlive = false;
        // Loops through all slots on the board and checks for both the black and white king
        for (int boardRow = 0; boardRow <= 7; boardRow++) {
            for (int boardCol = 0; boardCol <= 7; boardCol++) {
                tempPiece = getPiece(boardRow, boardCol);
                if (tempPiece != null) {
                    switch (tempPiece.getCharacter()) {
                        // King chars
                        case '\u265a':
                            blackKingAlive = true;
                            break;
                        case '\u2654':
                            whiteKingAlive = true;
                            break;
                    }
                }
            }
        }
        // returns true if either black or white king is missing
        return !(whiteKingAlive && blackKingAlive);
    }

    // Constructs a String that represents the Board object's 2D array. Returns
    // the fully constructed String.
    public String toString() {
        // creates the first row to the board. This is the numbers in the top row
        String boardString = "\u3000\u2001\uFF10\uFF11\uFF12\uFF13\uFF14\uFF15\uFF16\uFF17\n";
        Piece tempPiece = null;
        // loops through adding the row numbers and the piece characters to the board. If a slot is empty, a space character is placed instead.
        for (int boardRow = 0; boardRow <= 7; boardRow++) {
            boardString += unicodeNumbers[boardRow] + "|";
            for (int boardCol = 0; boardCol <= 7; boardCol++) {
                tempPiece = getPiece(boardRow, boardCol);
                if (tempPiece == null) boardString += "\u2001|";
                else boardString += tempPiece.getCharacter() + "|";
            }
            boardString += "\n";
        }
        return boardString;
    }

    // Sets every cell of the array to null. For debugging and grading purposes.
    public void clear() {
        // Loops through every board position and changes the position to null
        for (int boardRow = 0; boardRow < board.length; boardRow++) {
            for (int boardCol = 0; boardCol < board[0].length; boardCol++) {
                setPiece(boardRow, boardCol, null);
            }
        }
    }


    // Ensures that the player's chosen move is even remotely legal.
    // Returns a boolean to signify whether:
    // - 'start' and 'end' fall within the array's bounds.
    // - 'start' contains a Piece object, i.e., not null.
    // - Player's color and color of 'start' Piece match.
    // - 'end' contains either no Piece or a Piece of the opposite color.
    public boolean verifySourceAndDestination(int startRow, int startCol, int endRow, int endCol, boolean isBlack) {
        // Checks if the start and end destinations are on the board. Returns false if not
        if (startRow >= board.length || startCol >= board[0].length || endRow >= board.length || endCol >= board[0].length) {
            return false;
        }
        if (Math.min(startRow, startCol) < 0 || Math.min(endRow, endCol) < 0) return false;
        // gets the piece that is trying to be moved
        Piece pieceAtStart = getPiece(startRow, startCol);
        // checks to make sure the selected position has a piece
        if (pieceAtStart == null) return false;
        // checks to make sure the selected piece is the same color as the current player
        if (pieceAtStart.getIsBlack() != isBlack) return false;
        // gets the piece at the position trying to be moved to
        Piece pieceAtEnd = getPiece(endRow, endCol);
        // if the end position is empty, the piece can be moved
        if (pieceAtEnd == null) return true;
        // checks if the end piece is the same color as the current player. If it is, the piece can't be moved to the position.
        if (pieceAtEnd.getIsBlack() == isBlack) return false;
        // if passed all other checks, the final position is valid
        return true;
    }

    // Check whether the 'start' position and 'end' position are adjacent to each other
    public boolean verifyAdjacent(int startRow, int startCol, int endRow, int endCol) {
        // checks if the start and end are within one square
        if (Math.abs(startRow - endRow) <= 1 && Math.abs(startCol - endCol) <= 1) {
            return true;
        }
        return false;
    }

    // Checks whether a given 'start' and 'end' position are a valid horizontal move.
    // Returns a boolean to signify whether:
    // - The entire move takes place on one row.
    // - All spaces directly between 'start' and 'end' are empty, i.e., null.
    public boolean verifyHorizontal(int startRow, int startCol, int endRow, int endCol) {
        // checks to see if the move stays in the same row
        if (startRow != endRow) return false;
        int highestRow = Math.max(startCol, endCol);
        int lowestColumn = Math.min(startCol, endCol);
        // checks every column between the slots to make sure there are no pieces between the points.
        for (int boardColumn = lowestColumn + 1; boardColumn < highestRow; boardColumn++) {
            if (getPiece(startRow, boardColumn) != null) return false;
        }
        return true;
    }

    // Checks whether a given 'start' and 'end' position are a valid vertical move.
    // Returns a boolean to signify whether:
    // - The entire move takes place on one column.
    // - All spaces directly between 'start' and 'end' are empty, i.e., null.
    public boolean verifyVertical(int startRow, int startCol, int endRow, int endCol) {
        // checks to see if the move stays in the same column
        if (startCol != endCol) return false;
        int highestRow = Math.max(startRow, endRow);
        int lowestRow = Math.min(startRow, endRow);
        // checks every row between the slots to make sure there are no pieces between the points.
        for (int boardRow = lowestRow + 1; boardRow < highestRow; boardRow++) {
            if (getPiece(boardRow, startCol) != null) return false;
        }
        return true;
    }

    // Checks whether a given 'start' and 'end' position are a valid diagonal move.
    // Returns a boolean to signify whether:
    // - The path from 'start' to 'end' is diagonal... change in row and col.
    // - All spaces directly between 'start' and 'end' are empty, i.e., null.
    public boolean verifyDiagonal(int startRow, int startCol, int endRow, int endCol) {
        // checks if the move is diagonal
        if (Math.abs(startRow - endRow) != Math.abs(startCol - endCol)) return false;
        // Interval to step the row and column, checks and sets if the step should be negative
        int rowStepInterval = 1;
        int colStepInterval = 1;
        if (startRow > endRow) rowStepInterval = -1;
        if (startCol > endCol) colStepInterval = -1;
        Piece tempPiece = null;
        // loops through every diagonal space between the start and end
        for (int diagSpaceNum = 1; diagSpaceNum < Math.abs(startRow - endRow); diagSpaceNum++) {
            tempPiece = getPiece(startRow + rowStepInterval * diagSpaceNum, startCol + colStepInterval * diagSpaceNum);
            // if the slot is not empty returns false
            if (tempPiece != null) return false;
        }
        return true;
    }
}
